//-
// ==========================================================================
// Copyright 2017 Autodesk, Inc. All rights reserved.
//
// Use of this software is subject to the terms of the Autodesk
// license agreement provided at the time of installation or download,
// or which otherwise accompanies this software in either electronic
// or hard copy form.
// ==========================================================================
//+

// Example Plugin: captureViewRenderCmd.cpp
//
//
// This is a plugin to capture some of the modelPanel4's frames
// into render targets.
//
// loadPlugin captureViewRenderCmd;
// captureViewRender -captureFrame 1 10;
// captureViewRender -showFrame 6;
//
#include <maya/MApiNamespace.h>

#include <maya/MArgDatabase.h>
#include <maya/MFnPlugin.h>
#include <maya/MGlobal.h>
#include <maya/MPxCommand.h>
#include <maya/MSyntax.h>

#include <maya/MAnimControl.h>
#include <maya/MTime.h>

#include <maya/MRenderTargetManager.h>
#include <maya/MViewport2Renderer.h>

#include <vector>

#define kCaptureFrame "-cf"
#define kCaptureFrameLong "-captureFrame"

#define kShowFrame "-sf"
#define kShowFrameLong "-showFrame"

class MayaViewportRenderData
{
public:
    MayaViewportRenderData(const MTime& renderTime) : mRenderTime(renderTime)
    {
        MHWRender::MRenderer* theRenderer = MHWRender::MRenderer::theRenderer();
        if (theRenderer)
        {
            const MHWRender::MRenderTargetManager* targetManager =
                theRenderer->getRenderTargetManager();
            if (!targetManager)
                return;
            mTarget = targetManager->acquireRenderTargetFromScreen(
                MString("CapturedTarget") + mRenderTime.value());
        }
    }

    ~MayaViewportRenderData()
    {
        MHWRender::MRenderer* theRenderer = MHWRender::MRenderer::theRenderer();
        if (theRenderer)
        {
            const MHWRender::MRenderTargetManager* targetManager =
                theRenderer->getRenderTargetManager();
            if (targetManager)
            {
                targetManager->releaseRenderTarget(mTarget);
            }
        }
    }

    static void doRenderTask(MayaViewportRenderData* renderData)
    {
        if (!renderData)
            return;

        MHWRender::MRenderer* theRenderer = MHWRender::MRenderer::theRenderer();
        if (!theRenderer)
        {
            return;
        }
        MHWRender::MRenderTarget*& target = renderData->mTarget;
        if (!target)
        {
            return;
        }

        MAnimControl::setCurrentTime(renderData->mRenderTime);
        theRenderer->render("modelPanel4", &target, 1);
    }

    const MTime& renderTime() const
    {
        return mRenderTime;
    }

    const MHWRender::MRenderTarget* renderTarget() const
    {
        return mTarget;
    }

private:
    MHWRender::MRenderTarget* mTarget;
    MTime mRenderTime;
};

// global variables
std::vector<MayaViewportRenderData*> gRenderData;

class captureViewRenderCmd : public MPxCommand
{
public:
    captureViewRenderCmd()
    {
    }
    virtual ~captureViewRenderCmd()
    {
    }

    static void* creator()
    {
        return new captureViewRenderCmd;
    }

    static MSyntax newSyntax()
    {
        MSyntax syntax;

        syntax.addFlag(kCaptureFrame, kCaptureFrameLong,
                       MSyntax::kTime /*startTime*/,
                       MSyntax::kTime /*endTime*/);
        syntax.addFlag(kShowFrame, kShowFrameLong, MSyntax::kTime);

        syntax.enableQuery(false);
        syntax.enableEdit(false);

        return syntax;
    }

    virtual MStatus doIt(const MArgList&);
};

MStatus captureViewRenderCmd::doIt(const MArgList& args)
{

    MStatus status = MStatus::kFailure;

    MArgDatabase argData(syntax(), args, &status);
    if (!status)
        return status;

    if (argData.isFlagSet(kCaptureFrame))
    {
        // Remove all previous captured frames
        for (unsigned int i = 0; i < gRenderData.size(); i++)
        {
            delete gRenderData[i];
        }
        gRenderData.clear();

        MTime startTime;
        MTime endTime;
        argData.getFlagArgument(kCaptureFrame, 0, startTime);
        argData.getFlagArgument(kCaptureFrame, 1, endTime);

        for (MTime currentTime = startTime; currentTime <= endTime;
             currentTime++)
        {
            // do render tasks
            MayaViewportRenderData* newRenderTask =
                new MayaViewportRenderData(currentTime);
            gRenderData.push_back(newRenderTask);

            MayaViewportRenderData::doRenderTask(newRenderTask);
        }
    }
    else if (argData.isFlagSet(kShowFrame))
    {
        MTime time;
        argData.getFlagArgument(kShowFrame, 0, time);

        // Find the frame in the render data list
        unsigned int frameToShow = gRenderData.size();
        for (unsigned int i = 0; i < gRenderData.size(); i++)
        {
            if (time == gRenderData[i]->renderTime())
            {
                frameToShow = i;
                break;
            }
        }

        if (frameToShow == gRenderData.size())
        {
            MGlobal::displayError("No captured frame found!");
            return MS::kFailure;
        }

        // Copy the captured target to screen.
        MRenderer::theRenderer()->copyTargetToScreen(
            gRenderData[frameToShow]->renderTarget());
    }

    return MS::kSuccess;
}

MStatus initializePlugin(MObject _obj)
{
    MFnPlugin plugin(_obj, PLUGIN_COMPANY, "1.0");
    MStatus stat = plugin.registerCommand("captureViewRender",
                                          captureViewRenderCmd::creator,
                                          captureViewRenderCmd::newSyntax);
    if (!stat)
        stat.perror("registerCommand");
    return stat;
}

MStatus uninitializePlugin(MObject _obj)
{
    MFnPlugin plugin(_obj);
    MStatus stat = plugin.deregisterCommand("captureViewRender");
    if (!stat)
        stat.perror("deregisterCommand");

    // Remove all captured frames' render data
    for (unsigned int i = 0; i < gRenderData.size(); i++)
    {
        delete gRenderData[i];
    }
    gRenderData.clear();

    return stat;
}