#-
# ==========================================================================
# Copyright (C) 1995 - 2006 Autodesk, Inc. and/or its licensors.  All
# rights reserved.
#
# The coded instructions, statements, computer programs, and/or related
# material (collectively the "Data") in these files contain unpublished
# information proprietary to Autodesk, Inc. ("Autodesk") and/or its
# licensors, which is protected by U.S. and Canadian federal copyright
# law and by international treaties.
#
# The Data is provided for use exclusively by You. You have the right
# to use, modify, and incorporate this Data into other products for
# purposes authorized by the Autodesk software license agreement,
# without fee.
#
# The copyright notices in the Software and this entire statement,
# including the above license grant, this restriction and the
# following disclaimer, must be included in all copies of the
# Software, in whole or in part, and all derivative works of
# the Software, unless such copies or derivative works are solely
# in the form of machine-executable object code generated by a
# source language processor.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND.
# AUTODESK DOES NOT MAKE AND HEREBY DISCLAIMS ANY EXPRESS OR IMPLIED
# WARRANTIES INCLUDING, BUT NOT LIMITED TO, THE WARRANTIES OF
# NON-INFRINGEMENT, MERCHANTABILITY OR FITNESS FOR A PARTICULAR
# PURPOSE, OR ARISING FROM A COURSE OF DEALING, USAGE, OR
# TRADE PRACTICE. IN NO EVENT WILL AUTODESK AND/OR ITS LICENSORS
# BE LIABLE FOR ANY LOST REVENUES, DATA, OR PROFITS, OR SPECIAL,
# DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES, EVEN IF AUTODESK
# AND/OR ITS LICENSORS HAS BEEN ADVISED OF THE POSSIBILITY
# OR PROBABILITY OF SUCH DAMAGES.
#
# ==========================================================================
#+

########################################################################
# DESCRIPTION:
#
# Produces the dependency graph node "spFootPrintLocator"
# and "spFootPrintLocatorManip".
#   
# This example demonstrates how to use the Show Manip Tool with a user-defined
# manipulator. The user-defined manipulator corresponds to the foot print locator.
#
# To use this plug-in:
#	(1) Execute the footPrintManip.py script to create a foot print locator.
#	(2) Select the foot print and then click the Show Manip Tool. 
#
########################################################################

import maya.OpenMaya as OpenMaya
import maya.OpenMayaMPx as OpenMayaMPx
import maya.OpenMayaRender as OpenMayaRender
import maya.OpenMayaUI as OpenMayaUI

import math
import sys

kPluginLocatorTypeName = "spFootPrintLocator"
kPluginLocatorManipTypeName = "spFootPrintLocatorManip"

footPrintLocatorId = OpenMaya.MTypeId(0x8700C)
footPrintLocatorManipId = OpenMaya.MTypeId(0x8700D)

glRenderer = OpenMayaRender.MHardwareRenderer.theRenderer()
glFT = glRenderer.glFunctionTable()

sole = ( [ 0.00, 0.0, -0.70 ],
[  0.04, 0.0, -0.69 ],
[  0.09, 0.0, -0.65 ],
[  0.13, 0.0, -0.61 ],
[  0.16, 0.0, -0.54 ],
[  0.17, 0.0, -0.46 ],
[  0.17, 0.0, -0.35 ],
[  0.16, 0.0, -0.25 ],
[  0.15, 0.0, -0.14 ],
[  0.13, 0.0,  0.00 ],
[  0.00, 0.0,  0.00 ],
[ -0.13, 0.0,  0.00 ],
[ -0.15, 0.0, -0.14 ],
[ -0.16, 0.0, -0.25 ],
[ -0.17, 0.0, -0.35 ],
[ -0.17, 0.0, -0.46 ],
[ -0.16, 0.0, -0.54 ],
[ -0.13, 0.0, -0.61 ],
[ -0.09, 0.0, -0.65 ],
[ -0.04, 0.0, -0.69 ],
[ -0.00, 0.0, -0.70 ] )
 
heel = ( [  0.00, 0.0, 0.06 ],
[  0.13, 0.0,  0.06 ],
[  0.14, 0.0,  0.15 ],
[  0.14, 0.0,  0.21 ],
[  0.13, 0.0,  0.25 ],
[  0.11, 0.0,  0.28 ],
[  0.09, 0.0,  0.29 ],
[  0.04, 0.0,  0.30 ],
[  0.00, 0.0,  0.30 ],
[ -0.04, 0.0,  0.30 ],
[ -0.09, 0.0,  0.29 ],
[ -0.11, 0.0,  0.28 ],
[ -0.13, 0.0,  0.25 ],
[ -0.14, 0.0,  0.21 ],
[ -0.14, 0.0,  0.15 ],
[ -0.13, 0.0,  0.06 ],
[ -0.00, 0.0,  0.06 ] )


class footPrintLocatorManip(OpenMayaMPx.MPxManipContainer):
	def __init__(self):
		OpenMayaMPx.MPxManipContainer.__init__(self)
		self.fDistanceManip = OpenMaya.MDagPath()
		self.fNodePath = OpenMaya.MDagPath()

	def createChildren(self):
		try:
			startPoint = OpenMaya.MPoint(0.0, 0.0, 0.0)
			direction = OpenMaya.MVector(0.0, 1.0, 0.0)
			
			self.fDistanceManip = self.addDistanceManip("distanceManip", "distance")

			distanceManipFn = OpenMayaUI.MFnDistanceManip(self.fDistanceManip)
			distanceManipFn.setStartPoint(startPoint)
			distanceManipFn.setDirection(direction)
		except:
			sys.stderr.write("ERROR: footPrintLocatorManip.createChildren\n")
			raise

	def plugToManipConversion( manipIndex ):
		try:
			numData = OpenMaya.MFnNumericData()
			numDataObj = numData.create(OpenMaya.MFnNumericData.k3Double)
			
			vec = self.nodeTranslation()
			numData.setData3Double(vec.x, vec.y, vec.z)
			
			returnData = OpenMaya.MManipData(numDataObj)
		except:
			sys.stderr.write("ERROR: footPrintLocatorManip.plugToManipConversion\n")
			raise
		return returnData

	def connectToDependNode(self, node):
		try:
			dagNodeFn = OpenMaya.MFnDagNode(node)
			dagNodeFn.getPath(self.fNodePath)
			
			distanceManipFn = OpenMayaUI.MFnDistanceManip(self.fDistanceManip)
			nodeFn = OpenMaya.MFnDependencyNode(node)
			
			sizePlug = nodeFn.findPlug('size')
			distanceManipFn.connectToDistancePlug(sizePlug)

			self.finishAddingManips()
			OpenMayaMPx.MPxManipContainer.connectToDependNode(self, node)
		except:
			sys.stderr.write("ERROR: footPrintLocatorManip.connectToDependNode\n")
			raise

	def draw(self, view, path, style, status):
		OpenMayaMPx.MPxManipContainer.draw(self, view, path, style, status)
		
		view.beginGL()

		textPosVector = self.nodeTranslation()
		textPosPoint = OpenMaya.MPoint(textPosVector.x, textPosVector.y, textPosVector.z)
		
		view.drawText('Stretch Me!', textPosPoint, OpenMayaUI.M3dView.kLeft)		
		view.endGL()

	def startPointCallback(self, index):
		numData = OpenMaya.MFnNumericData()
		numDataObj = numData.create(OpenMaya.MFnNumericData.k3Double)
		
		vec = self.nodeTranslation()
		numData.setData3Double(vec.x, vec.y, vec.z)
		
		return OpenMayaUI.MManipData(numDataObj)

	def nodeTranslation(self):
		dagFn = OpenMaya.MFnDagNode(self.fNodePath)
		path = OpenMaya.MDagPath()
		
		dagFn.getPath(path)
		
		# pop from the shape to the transform
		path.pop()
		
		transformFn = OpenMaya.MFnTransform(path)
		return transformFn.translation(OpenMaya.MSpace.kWorld)


class footPrintLocator(OpenMayaMPx.MPxLocatorNode):
	size = OpenMaya.MObject()
	
	def __init__(self):
		OpenMayaMPx.MPxLocatorNode.__init__(self)
		
	def compute(self, plug, dataBlock):
		return OpenMaya.kUnknownParameter

	def draw(self, view, path, style, status):
		thisNode = self.thisMObject()

		plug = OpenMaya.MPlug(thisNode, self.size)
		
		sizeVal = plug.asMDistance()
		
		multiplier = sizeVal.asCentimeters()

		view.beginGL()

		if style == OpenMayaUI.M3dView.kFlatShaded or style == OpenMayaUI.M3dView.kGouraudShaded:
			glFT.glPushAttrib(OpenMayaRender.MGL_CURRENT_BIT)
			
			if status == OpenMayaUI.M3dView.kActive:
				view.setDrawColor( 13, OpenMayaUI.M3dView.kActiveColors )
			else:
				view.setDrawColor( 13, OpenMayaUI.M3dView.kDormantColors )

			last = len(sole) - 1
			glFT.glBegin( OpenMayaRender.MGL_TRIANGLE_FAN )
			for i in range(last):
				glFT.glVertex3f(sole[i][0]*multiplier, sole[i][1]*multiplier, sole[i][2]*multiplier)
			glFT.glEnd()

			last = len(heel) - 1
			glFT.glBegin( OpenMayaRender.MGL_TRIANGLE_FAN )
			for i in range(last):
				glFT.glVertex3f(heel[i][0]*multiplier, heel[i][1]*multiplier, heel[i][2]*multiplier)
			glFT.glEnd()

			glFT.glPopAttrib()

		glFT.glBegin(OpenMayaRender.MGL_LINES)
		
		last = len(sole) - 1
		for i in range(last):
			glFT.glVertex3f( sole[i][0]*multiplier, sole[i][1]*multiplier, sole[i][2]*multiplier )
			glFT.glVertex3f( sole[i+1][0]*multiplier, sole[i+1][1]*multiplier, sole[i+1][2]*multiplier )
			
		last = len(heel) - 1
		for i in range(last):
			glFT.glVertex3f( heel[i][0]*multiplier, heel[i][1]*multiplier, heel[i][2]*multiplier )
			glFT.glVertex3f( heel[i+1][0]*multiplier, heel[i+1][1]*multiplier, heel[i+1][2]*multiplier )
		
		glFT.glEnd()

		view.endGL()

	def isBounded(self):
		return True

	def boundingBox(self):
		thisNode = self.thisMObject()
		plug = OpenMaya.MPlug(thisNode, self.size)
		
		sizeVal = plug.asMDistance()
		
		multiplier = sizeVal.asCentimeters()
		
		corner1 = OpenMaya.MPoint(-0.17, 0.0, -0.7)
		corner2 = OpenMaya.MPoint(0.17, 0.0, 0.3)
		
		corner1 = corner1 * multiplier
		corner2 = corner2 * multiplier
		
		bbox = OpenMaya.MBoundingBox( corner1, corner2 )
		return bbox


def locatorCreator():
	return OpenMayaMPx.asMPxPtr( footPrintLocator() )

def locatorInitializer():
	unitFn = OpenMaya.MFnUnitAttribute()
	footPrintLocator.size = unitFn.create("size", "in", OpenMaya.MFnUnitAttribute.kDistance)
	unitFn.setDefault(10.0)
	unitFn.setStorable(True)
	unitFn.setWritable(True)
	
	footPrintLocator.addAttribute( footPrintLocator.size )
	OpenMayaMPx.MPxManipContainer.addToManipConnectTable(footPrintLocatorId)

def locatorManipCreator():
	return OpenMayaMPx.asMPxPtr( footPrintLocatorManip() )

def locatorManipInitializer():
	OpenMayaMPx.MPxManipContainer.initialize()


# initialize the script plug-in
def initializePlugin(mobject):
	mplugin = OpenMayaMPx.MFnPlugin(mobject)

	try:
		mplugin.registerNode( kPluginLocatorTypeName, footPrintLocatorId, locatorCreator, locatorInitializer, OpenMayaMPx.MPxNode.kLocatorNode )
	except:
		sys.stderr.write( "Failed to register node: %s" % kPluginLocatorTypeName )
		raise

	try:
		mplugin.registerNode( kPluginLocatorManipTypeName, footPrintLocatorManipId, locatorManipCreator, locatorManipInitializer, OpenMayaMPx.MPxNode.kManipContainer )
	except:
		sys.stderr.write( "Failed to register node: %s" % kPluginLocatorManipTypeName )
		raise


# uninitialize the script plug-in
def uninitializePlugin(mobject):
	mplugin = OpenMayaMPx.MFnPlugin(mobject)
	try:
		mplugin.deregisterNode( footPrintLocatorId )
	except:
		sys.stderr.write( "Failed to deregister node: %s" % kPluginLocatorTypeName )
		raise
	
	try:
		mplugin.deregisterNode( footPrintLocatorManipId )
	except:
		sys.stderr.write( "Failed to deregister node: %s" % kPluginLocatorManipTypeName )
		raise
