
#-
# ==========================================================================
# Copyright (C) 1995 - 2006 Autodesk, Inc. and/or its licensors.  All 
# rights reserved.
#
# The coded instructions, statements, computer programs, and/or related 
# material (collectively the "Data") in these files contain unpublished 
# information proprietary to Autodesk, Inc. ("Autodesk") and/or its 
# licensors, which is protected by U.S. and Canadian federal copyright 
# law and by international treaties.
#
# The Data is provided for use exclusively by You. You have the right 
# to use, modify, and incorporate this Data into other products for 
# purposes authorized by the Autodesk software license agreement, 
# without fee.
#
# The copyright notices in the Software and this entire statement, 
# including the above license grant, this restriction and the 
# following disclaimer, must be included in all copies of the 
# Software, in whole or in part, and all derivative works of 
# the Software, unless such copies or derivative works are solely 
# in the form of machine-executable object code generated by a 
# source language processor.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND. 
# AUTODESK DOES NOT MAKE AND HEREBY DISCLAIMS ANY EXPRESS OR IMPLIED 
# WARRANTIES INCLUDING, BUT NOT LIMITED TO, THE WARRANTIES OF 
# NON-INFRINGEMENT, MERCHANTABILITY OR FITNESS FOR A PARTICULAR 
# PURPOSE, OR ARISING FROM A COURSE OF DEALING, USAGE, OR 
# TRADE PRACTICE. IN NO EVENT WILL AUTODESK AND/OR ITS LICENSORS 
# BE LIABLE FOR ANY LOST REVENUES, DATA, OR PROFITS, OR SPECIAL, 
# DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES, EVEN IF AUTODESK 
# AND/OR ITS LICENSORS HAS BEEN ADVISED OF THE POSSIBILITY 
# OR PROBABILITY OF SUCH DAMAGES.
#
# ==========================================================================
#+

#	Creation Date:   2 October 2006

########################################################################
# DESCRIPTION:
#
# Produces the Python command "spHelix". 
#
# This script creates a NURBS curve in the shape of a helix.
# The command accepts these two arguments:
#
#	p=#		The pitch of the helix, default to 0.5
#	r=#		The radius of the helix, default to 4.0
#
# Example:
#
#	From Python:
#		import maya
#		maya.cmds.spHelix(p=0.3, r=7)
#
#	From Mel:
#		spHelix -p 0.3 -r 7
#
########################################################################

import maya.OpenMaya as OpenMaya
import maya.OpenMayaMPx as OpenMayaMPx
import sys, math

kPluginCmdName="spHelix"

kPitchFlag = "-p"
kPitchLongFlag = "-pitch"
kRadiusFlag = "-r"
kRadiusLongFlag = "-radius"

# command
class scriptedCommand(OpenMayaMPx.MPxCommand):
	def __init__(self):
		OpenMayaMPx.MPxCommand.__init__(self)
	
	def doIt(self, args):
		deg = 3
		ncvs = 20
		spans = ncvs - deg
		nknots = spans+2*deg-1
		radius = 4.0
		pitch = 0.5
		
		# Parse the arguments.
		argData = OpenMaya.MArgDatabase(self.syntax(), args)
		if argData.isFlagSet(kPitchFlag):
			pitch = argData.flagArgumentDouble(kPitchFlag, 0)
		if argData.isFlagSet(kRadiusFlag):
			radius = argData.flagArgumentDouble(kRadiusFlag, 0)

		controlVertices = OpenMaya.MPointArray()
		knotSequences = OpenMaya.MDoubleArray()

		# Set up cvs and knots for the helix
		#
		for i in range(0, ncvs):
			controlVertices.append( OpenMaya.MPoint( radius * math.cos(i),
				pitch * i, radius * math.sin(i) ) )

		for i in range(0, nknots):
			knotSequences.append( i )
		
		# Now create the curve
		#
		curveFn = OpenMaya.MFnNurbsCurve()
		
		nullObj = OpenMaya.MObject()

		try:
			# This plugin normally creates the curve by passing in the
			# cv's.  A function to create curves by passing in the ep's
			# has been added.  Set this to False to get that behaviour.
			#
			if True:
				curveFn.create( controlVertices,
								knotSequences, deg, 
								OpenMaya.MFnNurbsCurve.kOpen, 
								0, 0, 
								nullObj )
			else:
				curveFn.createWithEditPoints(controlVertices,
								3, OpenMaya.MFnNurbsCurve.kOpen,
								False, False, False)
		except:
			sys.stderr.write( "Error creating curve.\n" )
			raise

# Creator
def cmdCreator():
	# Create the command
	return OpenMayaMPx.asMPxPtr( scriptedCommand() )

# Syntax creator
def syntaxCreator():
	syntax = OpenMaya.MSyntax()
	syntax.addFlag(kPitchFlag, kPitchLongFlag, OpenMaya.MSyntax.kDouble)
	syntax.addFlag(kRadiusFlag, kRadiusLongFlag, OpenMaya.MSyntax.kDouble)
	return syntax

# Initialize the script plug-in
def initializePlugin(mobject):
	mplugin = OpenMayaMPx.MFnPlugin(mobject, "Autodesk", "1.0", "Any")
	try:
		mplugin.registerCommand( kPluginCmdName, cmdCreator, syntaxCreator )
	except:
		sys.stderr.write( "Failed to register command: %s\n" % kPluginCmdName )
		raise

# Uninitialize the script plug-in
def uninitializePlugin(mobject):
	mplugin = OpenMayaMPx.MFnPlugin(mobject)
	try:
		mplugin.deregisterCommand( kPluginCmdName )
	except:
		sys.stderr.write( "Failed to unregister command: %s\n" % kPluginCmdName )
		raise

	
