########################################################################
# DESCRIPTION:
#
# Produces the Python command "spWhatIs".   
#
# This simple command prints a message describing the API types of Maya objects.
# If no Maya objects are passed to the command, then it lists the types of all
# of the currently selected objects.
#
# For each object, the following information will be printed:
#
#	- name of the object 
#	- API type for the object 
#	- API function sets that could be used on the object.
#	  Note that not every function set listed actually exists. 
#
# This list is essentially the class derivation list containing all parent classes of this object.
#
# To use, make sure that whatIsCmd.py is in your MAYA_PLUG_IN_PATH (and the C++
# version is not) then do the following: 
#
#	(1) Create a NURBS sphere and make sure that the sphere is selected.
#	(2) Enter the following commands:
#		
#		import maya.cmds
#		maya.cmds.loadPlugin("whatIsCmd.py")
#		maya.cmds.spWhatIs()
#
# The command results in the following output in the script editor: 
#
#		Name: nurbsSphere1 
#		
#		Type: kTransform 
#
#		Function Sets: kBase, kNamedObject, kDependencyNode, kDagNode , kTransform 
#
########################################################################

import sys
import maya.OpenMaya as OpenMaya
import maya.OpenMayaMPx as OpenMayaMPx


# command
class WhatIsCmd(OpenMayaMPx.MPxCommand):
	kPluginCmdName = "spWhatIs"

	def __init__(self):
		OpenMayaMPx.MPxCommand.__init__(self)

	@staticmethod
	def cmdCreator():
		return OpenMayaMPx.asMPxPtr( WhatIsCmd() )

	def doIt(self, args):
		selectList = OpenMaya.MSelectionList()

		OpenMaya.MGlobal.getActiveSelectionList( selectList )

		node = OpenMaya.MObject()
		depFn = OpenMaya.MFnDependencyNode()

		iter = OpenMaya.MItSelectionList(selectList)

		while (iter.isDone() == 0):
			iter.getDependNode( node )

			depFn.setObject(node)

			name = depFn.name()
			types = []
			OpenMaya.MGlobal.getFunctionSetList( node, types )

			print "Name: %s" % name
			print "Type: %s" % node.apiTypeStr()
			sys.stdout.write( "Function Sets: " )
			sys.stdout.write(", ".join(types) + '\n')

			iter.next()


# Initialize the script plug-in
def initializePlugin(plugin):
	pluginFn = OpenMayaMPx.MFnPlugin(plugin)
	try:
		pluginFn.registerCommand(
			WhatIsCmd.kPluginCmdName, WhatIsCmd.cmdCreator
		)
	except:
		sys.stderr.write(
			"Failed to register command: %s\n" % WhatIsCmd.kPluginCmdName
		)
		raise

# Uninitialize the script plug-in
def uninitializePlugin(plugin):
	pluginFn = OpenMayaMPx.MFnPlugin(plugin)
	try:
		pluginFn.deregisterCommand(WhatIsCmd.kPluginCmdName)
	except:
		sys.stderr.write(
			"Failed to unregister command: %s\n" % WhatIsCmd.kPluginCmdName
		)
		raise

#-
# ==========================================================================
# Copyright (C) 2011 Autodesk, Inc. and/or its licensors.  All
# rights reserved.
#
# The coded instructions, statements, computer programs, and/or related
# material (collectively the "Data") in these files contain unpublished
# information proprietary to Autodesk, Inc. ("Autodesk") and/or its
# licensors, which is protected by U.S. and Canadian federal copyright
# law and by international treaties.
#
# The Data is provided for use exclusively by You. You have the right
# to use, modify, and incorporate this Data into other products for
# purposes authorized by the Autodesk software license agreement,
# without fee.
#
# The copyright notices in the Software and this entire statement,
# including the above license grant, this restriction and the
# following disclaimer, must be included in all copies of the
# Software, in whole or in part, and all derivative works of
# the Software, unless such copies or derivative works are solely
# in the form of machine-executable object code generated by a
# source language processor.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND.
# AUTODESK DOES NOT MAKE AND HEREBY DISCLAIMS ANY EXPRESS OR IMPLIED
# WARRANTIES INCLUDING, BUT NOT LIMITED TO, THE WARRANTIES OF
# NON-INFRINGEMENT, MERCHANTABILITY OR FITNESS FOR A PARTICULAR
# PURPOSE, OR ARISING FROM A COURSE OF DEALING, USAGE, OR
# TRADE PRACTICE. IN NO EVENT WILL AUTODESK AND/OR ITS LICENSORS
# BE LIABLE FOR ANY LOST REVENUES, DATA, OR PROFITS, OR SPECIAL,
# DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES, EVEN IF AUTODESK
# AND/OR ITS LICENSORS HAS BEEN ADVISED OF THE POSSIBILITY
# OR PROBABILITY OF SUCH DAMAGES.
#
# ==========================================================================
#+
