"""
Custom file resolver derived from MPxFileResolver that handles
the URI 'file' scheme.
When this resolver is active, URI file paths using the 'file:///'
scheme will be processed using methods on this class.
Refer to MPxFileResolver for more information about custom file resolvers.

To use, make sure that adskFileSchemeResolver.py is in
your MAYA_PLUG_IN_PATH then do the following:

# Load the plug-in
import maya.cmds
maya.cmds.loadPlugin("adskFileSchemeResolver.py")
# Once loaded, Maya will call the resolver methods in this plug-in when
# a URI file path is encountered during file resolution processing
# (file open, setAttr, etc.)
# Unload the plug-in
maya.cmds.unloadPlugin("adskFileSchemeResolver")
# Maya will no longer have access to this file
# resolver to handle URI file paths using the 'file:///' scheme  

"""

import sys
import maya.OpenMaya as OpenMaya
import maya.OpenMayaMPx as OpenMayaMPx
import maya.cmds as cmds


# 'file' scheme resolver 
class adskFileSchemeResolver(OpenMayaMPx.MPxFileResolver):
	"""
	This custom plug-in resolver handles the 'file' uri scheme.
	The resolution algorithm will return the fully qualified file path
	obtained from the URI directory and filename elements. 
	"""

	kPluginURIScheme = "file"
	kPluginResolverName = "adskFileSchemeResolver"

	def __init__(self):
		OpenMayaMPx.MPxFileResolver.__init__(self)

	def uriScheme(self):
		return(self.kPluginURIScheme)

	def resolveURI(self,uriFilePath,mode):
		# Retrieve the path (directory and filename) from the URI
		# Resolved path is this fully qualified path
		# Note: This resolver does not need to use the resolver mode
		resolvedFilePath = uriFilePath.getPath()
		return resolvedFilePath

	@staticmethod
	# Creator for the proxy instance 
	def theCreator():
		return OpenMayaMPx.asMPxPtr( adskFileSchemeResolver() )

	@staticmethod
	def className():
		return 'adskFileSchemeResolver'



# Initialize the script plug-in
def initializePlugin(plugin):
	pluginFn = OpenMayaMPx.MFnPlugin(plugin)
	try:
		pluginFn.registerURIFileResolver( adskFileSchemeResolver.kPluginResolverName,
						  adskFileSchemeResolver.kPluginURIScheme,
						  adskFileSchemeResolver.theCreator )
	except:
		sys.stderr.write( "Failed to register custom resolver: %s for scheme: %s\n" %
				  (adskFileSchemeResolver.kPluginResolverName,
				   adskFileSchemeResolver.kPluginURIScheme ))
		raise

# Uninitialize the script plug-in
def uninitializePlugin(plugin):
	pluginFn = OpenMayaMPx.MFnPlugin(plugin)
	try:
		pluginFn.deregisterURIFileResolver(adskFileSchemeResolver.kPluginResolverName)
	except:
		sys.stderr.write(
			"Failed to deregister custom file resolver: %s\n" %
			adskFileSchemeResolver.kPluginResolverName)
		raise

#-
# ==========================================================================
# Copyright (C) 2012 Autodesk, Inc. and/or its licensors.  All
# rights reserved.
#
# The coded instructions, statements, computer programs, and/or related
# material (collectively the "Data") in these files contain unpublished
# information proprietary to Autodesk, Inc. ("Autodesk") and/or its
# licensors, which is protected by U.S. and Canadian federal copyright
# law and by international treaties.
#
# The Data is provided for use exclusively by You. You have the right
# to use, modify, and incorporate this Data into other products for
# purposes authorized by the Autodesk software license agreement,
# without fee.
#
# The copyright notices in the Software and this entire statement,
# including the above license grant, this restriction and the
# following disclaimer, must be included in all copies of the
# Software, in whole or in part, and all derivative works of
# the Software, unless such copies or derivative works are solely
# in the form of machine-executable object code generated by a
# source language processor.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND.
# AUTODESK DOES NOT MAKE AND HEREBY DISCLAIMS ANY EXPRESS OR IMPLIED
# WARRANTIES INCLUDING, BUT NOT LIMITED TO, THE WARRANTIES OF
# NON-INFRINGEMENT, MERCHANTABILITY OR FITNESS FOR A PARTICULAR
# PURPOSE, OR ARISING FROM A COURSE OF DEALING, USAGE, OR
# TRADE PRACTICE. IN NO EVENT WILL AUTODESK AND/OR ITS LICENSORS
# BE LIABLE FOR ANY LOST REVENUES, DATA, OR PROFITS, OR SPECIAL,
# DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES, EVEN IF AUTODESK
# AND/OR ITS LICENSORS HAS BEEN ADVISED OF THE POSSIBILITY
# OR PROBABILITY OF SUCH DAMAGES.
#
# ==========================================================================
#+
