########################################################################
# DESCRIPTION:
#
# Produces the Python command "pyWhatIs".
#
# This simple command prints a message describing the API types of Maya objects.
# If no Maya objects are passed to the command, then it lists the types of
# all of the currently selected objects.
#
# To use, make sure that pyWhatIsCmd.py is in your MAYA_PLUG_IN_PATH,
# then do the following:
#
#	import maya.cmds
#	maya.cmds.loadPlugin("pyWhatIsCmd.py")
#	maya.cmds.pyWhatIs()
#
# For each object, the following information will be printed: 
#	- name of the object 
#	- API type for the object 
#	- API function sets that can be used on the object. Note that not every
#	  function set listed actually exists.
#
# This list is essentially the class derivation list containing all parent classes of this object.
#
# For example, create a NURBS sphere, make sure the sphere is selected and enter the command:
#
#	maya.cmds.pyWhatIs()
#
# The command results in the following output in the script editor: 
#
#	Name: nurbsSphere1 
#
#	Type: kTransform 
#
#	Function Sets: kBase, kNamedObject, kDependencyNode, kDagNode , kTransform 
#
########################################################################

import sys
import maya.api.OpenMaya as om

def maya_useNewAPI():
	"""
	The presence of this function tells Maya that the plugin produces, and
	expects to be passed, objects created using the Maya Python API 2.0.
	"""
	pass


# command
class PyWhatIsCmd(om.MPxCommand):
	kPluginCmdName = "pyWhatIs"

	def __init__(self):
		om.MPxCommand.__init__(self)

	@staticmethod
	def cmdCreator():
		return PyWhatIsCmd()

	def doIt(self, args):
		selectList = om.MGlobal.getActiveSelectionList()
		depFn = om.MFnDependencyNode()

		for i in range(selectList.length()):
			node = selectList.getDependNode(i)

			depFn.setObject(node)

			types = om.MGlobal.getFunctionSetList(node)

			print "Name: %s" % depFn.name()
			print "Type: %s" % node.apiTypeStr
			sys.stdout.write( "Function Sets: " )
			sys.stdout.write(", ".join(types) + '\n')


# Initialize the script plug-in
def initializePlugin(plugin):
	pluginFn = om.MFnPlugin(plugin)
	try:
		pluginFn.registerCommand(
			PyWhatIsCmd.kPluginCmdName, PyWhatIsCmd.cmdCreator
		)
	except:
		sys.stderr.write(
			"Failed to register command: %s\n" % PyWhatIsCmd.kPluginCmdName
		)
		raise

# Uninitialize the script plug-in
def uninitializePlugin(plugin):
	pluginFn = om.MFnPlugin(plugin)
	try:
		pluginFn.deregisterCommand(PyWhatIsCmd.kPluginCmdName)
	except:
		sys.stderr.write(
			"Failed to unregister command: %s\n" % PyWhatIsCmd.kPluginCmdName
		)
		raise

#-
# ==========================================================================
# Copyright (C) 2011 Autodesk, Inc. and/or its licensors.  All
# rights reserved.
#
# The coded instructions, statements, computer programs, and/or related
# material (collectively the "Data") in these files contain unpublished
# information proprietary to Autodesk, Inc. ("Autodesk") and/or its
# licensors, which is protected by U.S. and Canadian federal copyright
# law and by international treaties.
#
# The Data is provided for use exclusively by You. You have the right
# to use, modify, and incorporate this Data into other products for
# purposes authorized by the Autodesk software license agreement,
# without fee.
#
# The copyright notices in the Software and this entire statement,
# including the above license grant, this restriction and the
# following disclaimer, must be included in all copies of the
# Software, in whole or in part, and all derivative works of
# the Software, unless such copies or derivative works are solely
# in the form of machine-executable object code generated by a
# source language processor.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND.
# AUTODESK DOES NOT MAKE AND HEREBY DISCLAIMS ANY EXPRESS OR IMPLIED
# WARRANTIES INCLUDING, BUT NOT LIMITED TO, THE WARRANTIES OF
# NON-INFRINGEMENT, MERCHANTABILITY OR FITNESS FOR A PARTICULAR
# PURPOSE, OR ARISING FROM A COURSE OF DEALING, USAGE, OR
# TRADE PRACTICE. IN NO EVENT WILL AUTODESK AND/OR ITS LICENSORS
# BE LIABLE FOR ANY LOST REVENUES, DATA, OR PROFITS, OR SPECIAL,
# DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES, EVEN IF AUTODESK
# AND/OR ITS LICENSORS HAS BEEN ADVISED OF THE POSSIBILITY
# OR PROBABILITY OF SUCH DAMAGES.
#
# ==========================================================================
#+
